{{-- Push Notification Script --}}
@php
    $pushEnabled = gs('pn') ?? false;
    $firebaseConfig = gs('firebase_config');
@endphp

@if($pushEnabled && $firebaseConfig && !empty($firebaseConfig->apiKey))
<script src="https://www.gstatic.com/firebasejs/8.10.1/firebase-app.js"></script>
<script src="https://www.gstatic.com/firebasejs/8.10.1/firebase-messaging.js"></script>

<script nonce="{{ csp_nonce() }}">
    "use strict";

    const firebaseConfig = @json($firebaseConfig);
    const isAuthenticated = {{ auth()->check() ? 'true' : 'false' }};

    // Initialize Firebase
    if (firebaseConfig && firebaseConfig.apiKey) {
        firebase.initializeApp({
            apiKey: firebaseConfig.apiKey,
            authDomain: firebaseConfig.authDomain,
            projectId: firebaseConfig.projectId,
            storageBucket: firebaseConfig.storageBucket,
            messagingSenderId: firebaseConfig.messagingSenderId,
            appId: firebaseConfig.appId,
            measurementId: firebaseConfig.measurementId
        });

        const messaging = firebase.messaging();

        // Check notification permission
        function checkNotificationPermission() {
            if (!('Notification' in window)) {
                console.log('Push notifications not supported');
                return;
            }

            if (Notification.permission === 'denied') {
                console.log('Push notifications blocked by user');
                return;
            }

            if (Notification.permission === 'default' && isAuthenticated) {
                // Show notification prompt banner
                showNotificationBanner();
            }

            if (Notification.permission === 'granted' && isAuthenticated) {
                initFirebaseMessaging();
            }
        }

        // Show notification permission banner
        function showNotificationBanner() {
            const banner = document.createElement('div');
            banner.id = 'push-notification-banner';
            banner.className = 'fixed bottom-4 left-4 right-4 md:left-auto md:right-4 md:w-96 z-50';
            banner.innerHTML = `
                <div class="card-dynamic rounded-2xl p-4 shadow-lg" style="border: 1px solid var(--border);">
                    <div class="flex items-start gap-3">
                        <div class="flex-shrink-0 w-10 h-10 rounded-xl flex items-center justify-center" style="background: var(--primary-light);">
                            <i class="ph ph-bell-ringing text-xl" style="color: var(--primary);"></i>
                        </div>
                        <div class="flex-1">
                            <p class="font-medium text-sm" style="color: var(--text-primary);">@lang('Enable Notifications')</p>
                            <p class="text-xs mt-0.5" style="color: var(--text-secondary);">@lang('Get notified about tickets, invoices, and service updates.')</p>
                            <div class="flex gap-2 mt-3">
                                <button onclick="requestNotificationPermission()" class="btn-primary px-3 py-1.5 text-xs rounded-lg">@lang('Enable')</button>
                                <button onclick="dismissNotificationBanner()" class="btn-secondary px-3 py-1.5 text-xs rounded-lg">@lang('Not Now')</button>
                            </div>
                        </div>
                        <button onclick="dismissNotificationBanner()" class="text-lg" style="color: var(--text-muted);">
                            <i class="ph ph-x"></i>
                        </button>
                    </div>
                </div>
            `;
            document.body.appendChild(banner);
        }

        // Dismiss notification banner
        window.dismissNotificationBanner = function() {
            const banner = document.getElementById('push-notification-banner');
            if (banner) {
                banner.remove();
            }
            // Store dismissal in localStorage
            localStorage.setItem('push_notification_dismissed', Date.now());
        }

        // Request notification permission
        window.requestNotificationPermission = function() {
            Notification.requestPermission().then(permission => {
                dismissNotificationBanner();
                if (permission === 'granted') {
                    initFirebaseMessaging();
                }
            });
        }

        // Initialize Firebase Messaging
        function initFirebaseMessaging() {
            messaging.getToken({ vapidKey: firebaseConfig.vapidKey || '' })
                .then(token => {
                    if (token) {
                        saveDeviceToken(token);
                    }
                })
                .catch(err => {
                    console.log('Failed to get FCM token:', err);
                });

            // Handle foreground messages
            messaging.onMessage(payload => {
                const { title, body, icon, click_action } = payload.notification || {};
                const data = payload.data || {};

                // Show browser notification
                if (Notification.permission === 'granted') {
                    const notification = new Notification(title || 'New Notification', {
                        body: body,
                        icon: data.icon || '/favicon.ico',
                        badge: '/favicon.ico',
                        vibrate: [200, 100, 200],
                        data: { click_action: data.click_action || click_action }
                    });

                    notification.onclick = function(event) {
                        event.preventDefault();
                        if (this.data.click_action) {
                            window.open(this.data.click_action, '_blank');
                        }
                        notification.close();
                    };
                }

                // Also show toast notification
                if (window.showToast) {
                    window.showToast(body || title, 'info');
                }
            });
        }

        // Save device token to server
        function saveDeviceToken(token) {
            fetch('{{ route("client.device.token") }}', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({ token: token, is_app: false })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    console.log('Device token saved');
                }
            })
            .catch(err => console.log('Failed to save device token:', err));
        }

        // Check if banner was recently dismissed
        const dismissed = localStorage.getItem('push_notification_dismissed');
        const dismissedRecently = dismissed && (Date.now() - parseInt(dismissed)) < 86400000; // 24 hours

        if (!dismissedRecently) {
            checkNotificationPermission();
        } else if (Notification.permission === 'granted' && isAuthenticated) {
            initFirebaseMessaging();
        }
    }
</script>
@endif

