{{--
    Theme Controller — Single Source of Truth
    ==========================================
    Prevents FOUC (Flash of Unstyled Content) by applying the correct
    dark/light class to <html> BEFORE the first paint.

    Used by:  layouts/client.blade.php  (client area)
              templates/*/layouts/app.blade.php  (frontend templates)

    How it works:
      Phase 1: Runs IMMEDIATELY (before DOM renders) — reads theme.json
               config via server-injected values, checks localStorage,
               applies html.dark class and data-theme attribute.
      Phase 2: Registers Alpine.js store on alpine:init so components
               can toggle theme reactively via $store.theme.toggle().

    Data flow:
      theme.json → templateMeta('supports.dark_mode') → JS configMode
      theme.json mtime → configId (invalidates localStorage on change)
      localStorage('theme') → user preference persistence
      window.__THEME_DARK__ → bridge between Phase 1 and Phase 2
--}}

{{-- Phase 1: Invincible Theme Controller v4.0 (Runs IMMEDIATELY — No Flash) --}}
<script nonce="{{ csp_nonce() }}">
    (function() {
        'use strict';

        function applyTheme() {
            try {
                // 1. Server-injected config from active template's theme.json
                var configMode = {!! json_encode(templateMeta('supports.dark_mode', 'system')) !!};
                var configId = "{{ file_exists(resource_path('views/templates/' . activeTemplateName() . '/theme.json')) ? filemtime(resource_path('views/templates/' . activeTemplateName() . '/theme.json')) : '0' }}";

                var saved = localStorage.getItem('theme');
                var savedV = localStorage.getItem('theme_v');
                var isDark = false;

                // 2. Config version sync — reset user preference if theme.json was modified
                //    (e.g. admin changed dark_mode from 'system' to true)
                if (savedV !== configId) {
                    saved = null;
                    localStorage.setItem('theme_v', configId);
                    localStorage.removeItem('theme');
                }

                // 3. Deterministic resolution
                if (saved !== null) {
                    isDark = (saved === 'dark');
                } else {
                    if (configMode === 'system') {
                        isDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
                    } else {
                        isDark = (configMode === true || configMode === 'true' || configMode == 1 || configMode == '1');
                    }
                }

                // 4. Apply to DOM — both class and data-attribute for CSS selector flexibility
                var doc = document.documentElement;
                if (isDark) {
                    doc.classList.add('dark');
                    doc.setAttribute('data-theme', 'dark');
                } else {
                    doc.classList.remove('dark');
                    doc.setAttribute('data-theme', 'light');
                }

                // 5. Bridge to Alpine.js and persist initial config-derived value
                window.__THEME_DARK__ = isDark;
                if (saved === null && configMode !== 'system') {
                    localStorage.setItem('theme', isDark ? 'dark' : 'light');
                }

            } catch (e) { console.error('[ThemeController] Error:', e); }
        }

        // Apply immediately on initial page load
        applyTheme();

        // Re-apply after Livewire SPA navigation (wire:navigate)
        document.addEventListener('livewire:navigating', applyTheme);
        document.addEventListener('livewire:navigated', applyTheme);
    })();
</script>

{{-- Phase 2: Alpine.js Reactive Theme Store --}}
<script nonce="{{ csp_nonce() }}">
    document.addEventListener('alpine:init', function() {
        Alpine.store('theme', {
            on: window.__THEME_DARK__ || false,

            toggle: function() {
                this.on = !this.on;
                localStorage.setItem('theme', this.on ? 'dark' : 'light');
                if (this.on) {
                    document.documentElement.classList.add('dark');
                    document.documentElement.setAttribute('data-theme', 'dark');
                } else {
                    document.documentElement.classList.remove('dark');
                    document.documentElement.setAttribute('data-theme', 'light');
                }
            }
        });
    });
</script>
