/**
 * Alpine.js Order Page Component
 * Handles pricing calculation and domain search.
 */
window.orderPage = function (config = {}) {
    return {
        selectedCycle: config.initialCycle || 'monthly',
        price: config.initialPrice || 0,

        // Domain Handling
        domainSelected: false,
        domainMode: 'register',
        domainQuery: config.initialDomain || '',
        loading: false,
        loadingMore: false,
        results: [],
        cursor: null,
        hasMore: false,
        existingDomain: '',
        domainEppCode: '',

        selectedDomainName: '',
        selectedDomainId: 0,
        selectedDomainPrice: 0,
        domainAction: 'register',

        // Addon Handling
        selectedAddons: [],
        addonPrices: {}, // Map of addonId -> price

        // Error & Loading States
        idProtection: false,
        idProtectionPrice: 0,
        errorMessage: '',
        searchPerformed: false,

        async performSearch() {
            if (!this.domainQuery.trim()) return;
            this.loading = true;
            this.results = [];
            this.cursor = null;
            this.errorMessage = '';
            this.searchPerformed = true;
            await this.fetchResults();
            this.loading = false;
        },

        async loadMore() {
            if (!this.hasMore || this.loadingMore) return;
            this.loadingMore = true;
            await this.fetchResults();
            this.loadingMore = false;
        },

        async fetchResults() {
            try {
                const response = await fetch('/api/domains/search', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({
                        query: this.domainQuery,
                        cursor: this.cursor,
                        limit: 45
                    })
                });

                if (!response.ok) throw new Error('Search request failed');

                const data = await response.json();
                if (data.success) {
                    this.results = [...this.results, ...data.data];
                    this.cursor = data.meta.next_cursor;
                    this.hasMore = data.meta.has_more;
                } else {
                    this.errorMessage = data.message || 'Domain search failed. Please try again.';
                }
            } catch (error) {
                console.error('Domain search failed:', error);
                this.errorMessage = 'Network error or service unavailable. Check your connection.';
            }
        },

        selectDomain(res) {
            this.selectedDomainName = res.domain;
            this.selectedDomainId = res.id;
            this.selectedDomainPrice = parseFloat(res.pricing.registration);
            this.idProtectionPrice = parseFloat(res.pricing.id_protection) || 0;
            this.domainAction = this.domainMode === 'transfer' ? 'transfer' : 'register';
            this.domainSelected = true;
            this.idProtection = false; // Reset on change
        },

        useExistingDomain() {
            if (!this.existingDomain) return;
            this.selectedDomainName = this.existingDomain;
            this.selectedDomainId = 0;
            this.selectedDomainPrice = 0;
            this.domainAction = 'existing';
            this.domainSelected = true;
        },

        toggleAddon(id, price) {
            const index = this.selectedAddons.indexOf(id);
            if (index > -1) {
                this.selectedAddons.splice(index, 1);
                delete this.addonPrices[id];
            } else {
                this.selectedAddons.push(id);
                this.addonPrices[id] = parseFloat(price);
            }
        },

        get addonsTotal() {
            return Object.values(this.addonPrices).reduce((a, b) => a + b, 0);
        },

        // Password Strength
        password: '',
        get passwordStrength() {
            if (!this.password) return 0;
            let strength = 0;
            if (this.password.length >= 8) strength += 25;
            if (/[A-Z]/.test(this.password)) strength += 25;
            if (/[0-9]/.test(this.password)) strength += 25;
            if (/[^A-Za-z0-9]/.test(this.password)) strength += 25;
            return strength;
        }
    }
}
